<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class StatsController extends Controller
{
    private $statsApiUrl;

    public function __construct()
    {
        $this->statsApiUrl = config('services.prestashop.stats_api_url');
    }

    /**
     * Récupérer les statistiques avec cache
     */
    public function getStats(Request $request)
    {
        $range = $request->input('range', 'daily');
        $metrics = $request->input('metrics', 'revenue,orders');
        $points = $request->input('points', 30);

        // Clé de cache unique
        $cacheKey = "stats_{$range}_{$metrics}_{$points}";

        // Durée de cache : 5 minutes
        $cacheDuration = 300;

        try {
            $stats = Cache::remember($cacheKey, $cacheDuration, function () use ($range, $metrics, $points) {
                return $this->fetchStatsFromPrestaShop($range, $metrics, $points);
            });

            return response()->json([
                'success' => true,
                'data' => $stats,
                'cached' => Cache::has($cacheKey),
            ]);

        } catch (\Exception $e) {
            Log::error('Stats API Error', [
                'error' => $e->getMessage(),
                'range' => $range,
                'metrics' => $metrics,
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Erreur lors de la récupération des statistiques',
                'detail' => config('app.debug') ? $e->getMessage() : null,
            ], 500);
        }
    }

    /**
     * Récupérer les stats depuis PrestaShop/API externe
     */
    private function fetchStatsFromPrestaShop($range, $metrics, $points)
    {
        $url = rtrim($this->statsApiUrl, '/') . '/stats';

        $params = [
            'range' => $range,
            'metrics' => $metrics,
            'points' => $points,
        ];

        $response = Http::timeout(20)
            ->withHeaders([
                'Accept' => 'application/json',
                'Authorization' => 'Bearer ' . config('services.prestashop.ws_key'),
            ])
            ->get($url, $params);

        if (!$response->successful()) {
            throw new \Exception('Stats API returned status ' . $response->status());
        }

        return $response->json();
    }

    /**
     * Invalider le cache des stats
     */
    public function clearCache(Request $request)
    {
        // Vérifier que l'utilisateur est admin
        if ($request->user()->role !== 'admin') {
            return response()->json([
                'success' => false,
                'error' => 'Non autorisé',
            ], 403);
        }

        Cache::flush();

        return response()->json([
            'success' => true,
            'message' => 'Cache vidé avec succès',
        ]);
    }

    /**
     * Récupérer les KPIs principaux
     */
    public function getKpis(Request $request)
    {
        $cacheKey = 'kpis_latest';
        $cacheDuration = 600; // 10 minutes

        try {
            $kpis = Cache::remember($cacheKey, $cacheDuration, function () {
                return $this->calculateKpis();
            });

            return response()->json([
                'success' => true,
                'data' => $kpis,
            ]);

        } catch (\Exception $e) {
            Log::error('KPIs Calculation Error', [
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Erreur lors du calcul des KPIs',
            ], 500);
        }
    }

    /**
     * Calculer les KPIs depuis PrestaShop
     */
    private function calculateKpis()
    {
        // Appeler l'API PrestaShop pour récupérer les données
        $wsKey = config('services.prestashop.ws_key');
        $apiUrl = config('services.prestashop.api_url');

        // Exemple : récupérer les commandes récentes
        $ordersUrl = rtrim($apiUrl, '/') . '/orders';
        $ordersResponse = Http::timeout(20)->get($ordersUrl, [
            'ws_key' => $wsKey,
            'output_format' => 'JSON',
            'display' => 'full',
            'filter[date_add]' => '[' . now()->subDays(30)->format('Y-m-d') . ',*]',
        ]);

        $orders = $ordersResponse->json()['orders'] ?? [];

        // Calculer les KPIs
        $totalRevenue = 0;
        $totalOrders = count($orders);
        $uniqueCustomers = [];

        foreach ($orders as $order) {
            $totalRevenue += (float) ($order['total_paid'] ?? 0);
            $customerId = $order['id_customer'] ?? null;
            if ($customerId && !in_array($customerId, $uniqueCustomers)) {
                $uniqueCustomers[] = $customerId;
            }
        }

        $avgOrderValue = $totalOrders > 0 ? $totalRevenue / $totalOrders : 0;

        return [
            'meta' => [
                'total_revenue' => round($totalRevenue, 2),
                'total_orders' => $totalOrders,
                'unique_customers' => count($uniqueCustomers),
                'avg_order_value' => round($avgOrderValue, 2),
            ],
            'updated_at' => now()->toIso8601String(),
        ];
    }
}
